from reportlab.lib import colors
from reportlab.lib.enums import TA_LEFT, TA_CENTER
from reportlab.lib.styles import ParagraphStyle
from reportlab.lib.units import mm
from reportlab.platypus import TableStyle, Table, Paragraph


class InvoiceDatesBox(Table):
    """ Table for invoice dates """
    default_lang = 'en'

    # Column widths
    auto_sized_column_index = 0
    default_col_widths = [
        0 * mm,  # headers (0 value means auto-sized)
        20 * mm,  # values
    ]
    # Table settings
    default_table_style = [
        ('GRID', (0, 0), (-1, -1), 1, colors.dodgerblue),
        ('VALIGN', (0, 0), (-1, 0), 'MIDDLE'),
        ('BACKGROUND', (0, 0), (-1, -1), colors.white),
        ('LEFTPADDING', (0, 0), (-1, -1), 2),
        ('RIGHTPADDING', (0, 0), (-1, -1), 2),
    ]
    # Left column paragraph style
    default_left_column_style = ParagraphStyle(
        name=None,
        fontName='Helvetica',
        fontSize=10,
        autoLeading='max',
        textColor=colors.black,
        alignment=TA_LEFT
    )
    # Right column paragraph style
    default_right_column_style = ParagraphStyle(
        name=None,
        fontName='Helvetica',
        fontSize=10,
        autoLeading='max',
        textColor=colors.black,
        alignment=TA_CENTER
    )
    dictionary = {
        'lbl_date': {
            'en': 'Invoice date',
            'fr': 'Date facture',
            'es': 'Fecha factura',
            'ca': 'Data factura',
            'it': 'Data fattura',
            'pt': 'Data da fatura'
        },
        'lbl_due_date': {
            'en': 'Invoice due date',
            'fr': 'Échéance facture',
            'es': 'Vencimiento factura',
            'ca': 'Venciment factura',
            'it': 'Data scadenza fattura',
            'pt': 'Data vencimento fatura'
        },
    }

    def __init__(self, *args, **kwargs) -> None:
        self.context: dict = kwargs.pop('context', {})
        self.max_width: int = kwargs.pop('max_width')
        self.lang: str = kwargs.pop('lang', self.default_lang)

        self.left_column_style = self.default_left_column_style
        self.right_column_style = self.default_right_column_style

        kwargs['data'] = self.get_formatted_data()
        kwargs['repeatRows'] = 1
        kwargs['colWidths'] = self.default_col_widths
        kwargs['style'] = TableStyle(self.default_table_style)
        super().__init__(*args, **kwargs)

    def wrap(self, *args, **kwargs) -> tuple:
        """ All columns are fixed width, except one column that is auto-sized to fill page width """
        available_width, available_height = args
        max_width = self.max_width if self.max_width else available_width

        # calculates current total width
        total_width = 0.0
        for column_width in self._colWidths:
            total_width += column_width

        # for auto-sized-column: replace original width with new calculated width
        total_width -= self._colWidths[self.auto_sized_column_index]
        self._colWidths[self.auto_sized_column_index] = max_width - total_width

        return super().wrap(*args, **kwargs)

    def translate(self, item: dict) -> str:
        """ Text translator """
        if isinstance(item, str):
            return item

        if isinstance(item, dict):
            if self.lang in item:
                return item.get(self.lang)

            if self.default_lang in item:
                return item.get(self.default_lang)

        return '* no translation found *'

    def get_formatted_data(self) -> list:
        """ Format table data """
        formatted_rows = []

        data_date: str = self.context.get('date')
        data_due_date: str = self.context.get('due_date')

        lbl_date: str = self.translate(self.dictionary.get('lbl_date'))
        lbl_due_date: str = self.translate(self.dictionary.get('lbl_due_date'))

        formatted_rows.append([
            Paragraph(text=f'<strong>{lbl_date}</strong>', style=self.left_column_style),
            Paragraph(text=f'<strong>{data_date}</strong>', style=self.right_column_style)
        ])
        formatted_rows.append([
            Paragraph(text=f'{lbl_due_date}', style=self.left_column_style),
            Paragraph(text=f'{data_due_date}', style=self.right_column_style)
        ])

        return formatted_rows
