import datetime

from reportlab.lib import colors
from reportlab.lib.units import mm
from reportlab.platypus import Flowable


class TimestampBand(Flowable):
    default_lang = 'en'

    default_font_name = 'Helvetica'
    default_font_size = 8
    default_font_color = colors.blue

    default_stroke_color = colors.CMYKColor(cyan=0, magenta=0, yellow=0, black=0.1)
    default_fill_color = colors.CMYKColor(cyan=0, magenta=0, yellow=0, black=0)
    default_text_color = colors.black

    default_width = 100 * mm
    default_height = 15 * mm

    dictionary = {
        'timestamp_message': {
            'en': 'Document created on  {}  UTC TIME',
            'fr': 'Document créé le  {}  HEURE UTC',
            'es': 'Documento creado el  {}  HORA UTC',
            'ca': 'Document creat el  {}  HORA UTC',
            'it': 'Documento creato il  {}  ORA UTC',
            'pt': 'Documento criado em  {}  HORA UTC'
        }
    }

    def __init__(self, *args, **kwargs) -> None:
        super().__init__()
        self.context: dict = kwargs.pop('context', dict())
        self.max_width: float = kwargs.pop('max_width', self.default_width)
        self.max_height: float = kwargs.pop('max_height', self.default_height)
        self.lang: str = kwargs.pop('lang', self.default_lang)

        self.font_name = self.default_font_name
        self.font_size = self.default_font_size
        self.font_color = self.default_font_color

        self.stroke_color = self.default_stroke_color
        self.fill_color = self.default_fill_color

    def wrap(self, *args) -> tuple:
        return self.max_width, self.max_height

    def draw(self) -> None:
        self.canv.saveState()
        self.canv.rotate(90)

        # draw box outline
        self.canv.setFillColor(self.fill_color)
        self.canv.setStrokeColor(self.stroke_color)
        self.canv.rect(0, 0, self.max_width, self.max_height, fill=1)

        # draw message data
        px = self.max_width - 5 * mm
        py = 3 * mm
        self.canv.setFont(self.font_name, self.font_size)
        self.canv.setFillColor(self.font_color)
        self.canv.drawRightString(px, py, self.get_formatted_text())

        # restore canvas state
        self.canv.restoreState()

    def translate(self, item: dict) -> str:
        """ Text translator """
        if isinstance(item, str):
            return item

        if isinstance(item, dict):
            if self.lang in item:
                return item.get(self.lang)

            if self.default_lang in item:
                return item.get(self.default_lang)

        return '***'

    def get_formatted_text(self) -> str:
        timestamp = datetime.datetime.now()
        message: str = self.translate(self.dictionary.get('timestamp_message'))
        return message.format(timestamp)
