from reportlab.lib import colors
from reportlab.lib.units import mm
from reportlab.platypus import Flowable


class InvoiceTitleBox(Flowable):
    default_lang = 'en'

    default_title_font_name = 'Helvetica-Bold'
    default_title_font_size = 24
    default_title_font_color = colors.black

    default_id_font_name = 'Helvetica-Bold'
    default_id_font_size = 14
    default_id_font_color = colors.black

    default_stroke_color = colors.CMYKColor(cyan=0, magenta=0, yellow=0, black=0.1)
    default_fill_color = colors.CMYKColor(cyan=0, magenta=0, yellow=0, black=0.1)

    default_width = 100 * mm
    default_height = 50 * mm

    dictionary = {
        'lbl_title': {
            'en': 'Invoice',
            'fr': 'Facture',
            'es': 'Factura',
            'ca': 'Factura',
            'it': 'Fattura',
            'pt': 'Fatura'
        }
    }

    def __init__(self, *args, **kwargs) -> None:
        super().__init__()
        self.context: dict = kwargs.pop('context', dict())
        self.max_width: float = kwargs.pop('max_width', self.default_width)
        self.max_height: float = kwargs.pop('max_height', self.default_height)
        self.lang: str = kwargs.pop('lang', self.default_lang)

        self.title_font_name = self.default_title_font_name
        self.title_font_size = self.default_title_font_size
        self.title_font_color = self.default_title_font_color

        self.id_font_name = self.default_id_font_name
        self.id_font_size = self.default_id_font_size
        self.id_font_color = self.default_id_font_color

        self.stroke_color = self.default_stroke_color
        self.fill_color = self.default_fill_color

    def wrap(self, *args) -> tuple:
        return self.max_width, self.max_height

    def draw(self) -> None:
        self.canv.saveState()

        # draw title
        lbl_title: str = self.translate(self.dictionary.get('lbl_title'))
        px = self.max_width
        py = self.max_height - (self.title_font_size * 0.7)
        self.canv.setFont(self.title_font_name, self.title_font_size)
        self.canv.setFillColor(self.title_font_color)
        self.canv.drawRightString(px, py, f'{lbl_title}')

        # draw invoice id
        data_invoice_id: str = self.context.get('id')
        px = self.max_width
        py = self.max_height - (self.title_font_size * 1.4)
        self.canv.setFont(self.id_font_name, self.id_font_size)
        self.canv.setFillColor(self.id_font_color)
        self.canv.drawRightString(px, py, f'{data_invoice_id}')

        # restore canvas state
        self.canv.restoreState()

    def translate(self, item: dict) -> str:
        """ Text translator """
        if isinstance(item, str):
            return item

        if isinstance(item, dict):
            if self.lang in item:
                return item.get(self.lang)

            if self.default_lang in item:
                return item.get(self.default_lang)

        return '***'

    def get_formatted_text(self) -> str:
        vat_number = self.translate(self.context.get('vat_number'))
        name = self.translate(self.context.get('name'))
        address1 = self.translate(self.context.get('address1'))
        address2 = self.translate(self.context.get('address2'))
        zipcode = self.translate(self.context.get('zipcode'))
        city = self.translate(self.context.get('city'))
        country = self.translate(self.context.get('country'))

        lbl_vat_number = self.translate(self.dictionary.get('vat_number', {}))
        lbl_issuer = self.translate(self.dictionary.get('issuer', {}))

        return f'{lbl_issuer} | {name} | {address1}, {address2} | {zipcode}  {city} | {country} |  {lbl_vat_number}  {vat_number}'
