from reportlab.lib import colors
from reportlab.lib.enums import TA_LEFT, TA_CENTER, TA_RIGHT
from reportlab.lib.styles import ParagraphStyle
from reportlab.lib.units import mm
from reportlab.platypus import TableStyle, Table, Paragraph


class InvoiceItemsBox(Table):
    """ Table for invoice items """
    default_lang = 'en'

    table_col_widths = [
        0 * mm,  # description (0 value means auto-sized)
        25 * mm,  # quantity
        20 * mm,  # unit price
        30 * mm,  # tax base
        15 * mm,  # tax percentage
        25 * mm,  # tax value
    ]
    # Table settings
    table_style = [
        ('GRID', (0, 0), (-1, -1), 1, colors.dodgerblue),
        ('BACKGROUND', (0, 0), (-1, 0), colors.dodgerblue),
        ('VALIGN', (0, 0), (-1, 0), 'MIDDLE'),
        ('BACKGROUND', (0, 1), (-1, -1), colors.white),
        ('LEFTPADDING', (0, 0), (-1, -1), 2),
        ('RIGHTPADDING', (0, 0), (-1, -1), 2),
    ]
    # header text style
    default_header_style = ParagraphStyle(
        name=None,
        fontName='Helvetica',
        fontSize=8,
        autoLeading='max',
        textColor=colors.white,
        alignment=TA_CENTER,
    )
    # right align text style
    default_right_style = ParagraphStyle(
        name=None,
        fontName='Helvetica',
        fontSize=10,
        autoLeading='max',
        textColor=colors.black,
        alignment=TA_RIGHT,
    )
    # left align text style
    default_left_style = ParagraphStyle(
        name=None,
        fontName='Helvetica',
        fontSize=10,
        autoLeading='max',
        textColor=colors.black,
        alignment=TA_LEFT
    )
    # center align text style
    default_center_style = ParagraphStyle(
        name=None,
        fontName='Helvetica',
        fontSize=10,
        autoLeading='max',
        textColor=colors.black,
        alignment=TA_CENTER
    )
    dictionary = {
        'lbl_description': {
            'en': 'DESCRIPTION',
            'fr': 'DESCRIPTION',
            'es': 'DESCRIPCIÓN',
            'ca': 'DESCRIPCIÓ',
            'it': 'DESCRIZIONE',
            'pt': 'DESCRIÇÃO'
        },
        'lbl_quantity': {
            'en': 'QUANTITY',
            'fr': 'QUANTITÉ',
            'es': 'CANTIDAD',
            'ca': 'QUANTITAT',
            'it': 'QUANTITÀ',
            'pt': 'QUANTIDADE'
        },
        'lbl_unit_price': {
            'en': 'UNIT<br/>PRICE',
            'fr': 'PRIX<br/>UNITAIRE',
            'es': 'PRECIO<br/>UNITARIO',
            'ca': 'PREU<br/>UNITARI',
            'it': 'PREZZO<br/>UNITARIO',
            'pt': 'PREÇO<br/>UNITÁRIO'
        },
        'lbl_tax_base': {
            'en': 'TAX<br/>BASE',
            'fr': 'REVENU<br/>IMPOSABLE',
            'es': 'BASE<br/>IMPONIBLE',
            'ca': 'BASE<br/>IMPOSABLE',
            'it': 'BASE<br/>IMPONIBILE',
            'pt': 'BASE<br/>TRIBUTÁVEL'
        },
        'lbl_tax_percentage': {
            'en': 'VAT %',
            'fr': 'TVA %',
            'es': 'IVA %',
            'ca': 'IVA %',
            'it': 'IVA %',
            'pt': 'IVA %'
        },
        'lbl_tax_amount': {
            'en': 'VAT AMOUNT',
            'fr': 'MONTANT TVA',
            'es': 'IMPORTE IVA',
            'ca': 'IMPORT IVA',
            'it': 'IMPORTO IVA',
            'pt': 'MONTANTE IVA'
        },
    }

    def __init__(self, *args, **kwargs) -> None:
        self.lang: str = kwargs.pop('lang', 'en')
        raw_data: list = kwargs.pop('data', [])

        self.header_style = self.default_header_style
        self.right_style = self.default_right_style
        self.left_style = self.default_left_style
        self.center_style = self.default_center_style

        kwargs['data'] = self.get_formatted_data(raw_data)
        kwargs['repeatRows'] = 1
        kwargs['colWidths'] = self.table_col_widths
        kwargs['style'] = TableStyle(self.table_style)
        super().__init__(*args, **kwargs)

    def wrap(self, *args, **kwargs) -> tuple:
        """ All columns are fixed width, except one column that is auto-sized to fill page width """
        # This is the column to be auto-sized
        auto_sized_column_index = 0

        # available_width, available_size
        aw, ah = args
        total_w = 0.0
        for cw in self._colWidths:
            total_w += cw
        total_w -= self._colWidths[auto_sized_column_index]
        self._colWidths[auto_sized_column_index] = aw - total_w
        return super().wrap(*args, **kwargs)

    def translate(self, item: dict) -> str:
        """ Text translator """
        if isinstance(item, str):
            return item

        if isinstance(item, dict):
            if self.lang in item:
                return item.get(self.lang)

            if self.default_lang in item:
                return item.get(self.default_lang)

        return '* no translation found *'

    def get_formatted_data(self, raw_data: list) -> list:
        """ Format full table data """

        # append headers row
        formatted_rows = [self.create_header_row()]

        # append data rows
        for data in raw_data:
            description = self.translate(data.get('description'))
            quantity = data.get('quantity') or '-'
            units = self.translate(data.get('units'))
            unit_price = data.get('unit_price') or '-'
            tax_base = data.get('tax_base') or '-'
            tax_percentage = data.get('tax_percentage') or '-'
            tax_value = data.get('tax_value') or '-'

            row = [
                Paragraph(description, self.left_style),
                Paragraph(f'{quantity} {units}', self.center_style),
                Paragraph(str(unit_price), self.center_style),
                Paragraph(str(tax_base), self.center_style),
                Paragraph(str(tax_percentage), self.center_style),
                Paragraph(str(tax_value), self.center_style),
            ]
            formatted_rows.append(row)

        return formatted_rows

    def create_header_row(self) -> list:
        """ Format table headers """

        lbl_description = self.translate(self.dictionary.get('lbl_description'))
        lbl_quantity = self.translate(self.dictionary.get('lbl_quantity'))
        lbl_unit_price = self.translate(self.dictionary.get('lbl_unit_price'))
        lbl_tax_base = self.translate(self.dictionary.get('lbl_tax_base'))
        lbl_tax_percentage = self.translate(self.dictionary.get('lbl_tax_percentage'))
        lbl_tax_amount = self.translate(self.dictionary.get('lbl_tax_amount'))

        header = [
            Paragraph(f'<strong>{lbl_description}</strong>', self.header_style),
            Paragraph(f'<strong>{lbl_quantity}</strong>', self.header_style),
            Paragraph(f'<strong>{lbl_unit_price}</strong>', self.header_style),
            Paragraph(f'<strong>{lbl_tax_base}</strong>', self.header_style),
            Paragraph(f'<strong>{lbl_tax_percentage}</strong>', self.header_style),
            Paragraph(f'<strong>{lbl_tax_amount}</strong>', self.header_style)
        ]
        return header
