from reportlab.lib import colors
from reportlab.lib.units import mm
from reportlab.platypus import Flowable


class InvoiceIssuerBand(Flowable):
    default_lang = 'en'

    default_issuer_font_name = 'Helvetica'
    default_issuer_font_size = 8
    default_issuer_font_color = colors.blue

    default_page_num_font_name = 'Helvetica'
    default_page_num_font_size = 8
    default_page_num_font_color = colors.indianred

    default_stroke_color = colors.CMYKColor(cyan=0, magenta=0, yellow=0, black=0.1)
    default_fill_color = colors.CMYKColor(cyan=0, magenta=0, yellow=0, black=0.1)
    default_text_color = colors.black

    default_width = 100 * mm
    default_height = 15 * mm

    dictionary = {
        'vat_number': {
            'en': 'VAT number',
            'fr': 'Numéro de TVA',
            'es': 'CIF',
            'ca': 'CIF',
            'it': 'Numero IVA',
            'pt': 'Número de IVA'
        },
        'issuer': {
            'en': 'Invoice issuer',
            'fr': 'Émetteur de la facture',
            'es': 'Emisor de la factura',
            'ca': 'Emissor de la factura',
            'it': 'Emittente della fattura',
            'pt': 'Emissor da fatura'
        },
        'page': {
            'en': 'page',
            'fr': 'page',
            'es': 'página',
            'ca': 'pàgina',
            'it': 'pagina',
            'pt': 'página'
        }
    }

    def __init__(self, *args, **kwargs) -> None:
        super().__init__()
        self.context: dict = kwargs.pop('context', dict())
        self.max_width: float = kwargs.pop('max_width', self.default_width)
        self.max_height: float = kwargs.pop('max_height', self.default_height)
        self.lang: str = kwargs.pop('lang', self.default_lang)

        self.issuer_font_name = self.default_issuer_font_name
        self.issuer_font_size = self.default_issuer_font_size
        self.issuer_font_color = self.default_issuer_font_color

        self.page_num_font_name = self.default_page_num_font_name
        self.page_num_font_size = self.default_page_num_font_size
        self.page_num_font_color = self.default_page_num_font_color

        self.stroke_color = self.default_stroke_color
        self.fill_color = self.default_fill_color

    def wrap(self, *args) -> tuple:
        return self.max_width, self.max_height

    def draw(self) -> None:
        self.canv.saveState()
        self.canv.rotate(90)

        # draw box outline
        self.canv.setFillColor(self.fill_color)
        self.canv.setStrokeColor(self.stroke_color)
        self.canv.rect(0, 0, self.max_width, self.max_height, fill=1)

        # draw issuer data
        px = self.max_width - 5 * mm
        py = 3 * mm
        self.canv.setFont(self.issuer_font_name, self.issuer_font_size)
        self.canv.setFillColor(self.issuer_font_color)
        self.canv.drawRightString(px, py, self.get_formatted_text())

        # draw page number
        lbl_page = self.translate(self.dictionary.get('page', {}))
        px = 0 + 5 * mm
        py = 3 * mm
        self.canv.setFont(self.page_num_font_name, self.page_num_font_size)
        self.canv.setFillColor(self.page_num_font_color)
        page_number = self.canv.getPageNumber()
        self.canv.drawString(px, py, f'{lbl_page}  {page_number}')

        # restore canvas state
        self.canv.restoreState()

    def translate(self, item: dict) -> str:
        """ Text translator """
        if isinstance(item, str):
            return item

        if isinstance(item, dict):
            if self.lang in item:
                return item.get(self.lang)

            if self.default_lang in item:
                return item.get(self.default_lang)

        return '***'

    def get_formatted_text(self) -> str:
        vat_number = self.translate(self.context.get('vat_number'))
        name = self.translate(self.context.get('name'))
        address1 = self.translate(self.context.get('address1'))
        address2 = self.translate(self.context.get('address2'))
        zipcode = self.translate(self.context.get('zipcode'))
        city = self.translate(self.context.get('city'))
        country = self.translate(self.context.get('country'))

        lbl_vat_number = self.translate(self.dictionary.get('vat_number', {}))
        lbl_issuer = self.translate(self.dictionary.get('issuer', {}))

        return f'{lbl_issuer}  >>  {name} | {address1}, {address2} | {zipcode}  {city} | {country} |  {lbl_vat_number}  {vat_number}'
