from reportlab.lib import colors
from reportlab.lib.colors import CMYKColor
from reportlab.lib.enums import TA_LEFT
from reportlab.lib.styles import ParagraphStyle
from reportlab.lib.units import mm
from reportlab.platypus import Flowable, Paragraph

from .invoice_qr_code import InvoiceQrCode


class InvoiceCustomerBox(Flowable):
    default_lang = 'en'

    default_stroke_color = CMYKColor(cyan=0.0, magenta=0.0, yellow=0.0, black=0.4)
    default_fill_color = CMYKColor(cyan=0.0, magenta=0.0, yellow=0.0, black=0.0)
    default_font_name = 'Helvetica'
    default_font_size = 12

    # left align text style
    default_left_style = ParagraphStyle(
        name=None,
        fontName='Helvetica',
        fontSize=12,
        autoLeading='max',
        textColor=colors.black,
        alignment=TA_LEFT
    )

    def __init__(self, max_width: float=90.0 * mm, max_height: float=35.0 * mm, *args, **kwargs) -> None:
        super().__init__()
        self.context: dict = kwargs.pop('data', {})
        self.lang: str = kwargs.pop('lang', 'en')
        self.max_width = max_width
        self.max_height = max_height
        self.stroke_color = kwargs.get('strokecolor', self.default_stroke_color)
        self.fill_color = kwargs.get('fillcolor', self.default_fill_color)

    def translate(self, item: dict) -> str:
        """ Text translator """
        if isinstance(item, str):
            return item

        if isinstance(item, dict):
            if self.lang in item:
                return item.get(self.lang)

            if self.default_lang in item:
                return item.get(self.default_lang)

        return '* no translation found *'

    def format_data(self) -> str:
        vat_number = self.translate(self.context.get('vat_number'))
        name = self.translate(self.context.get('name'))
        address1 = self.translate(self.context.get('address1'))
        address2 = self.translate(self.context.get('address2'))
        zipcode = self.translate(self.context.get('zipcode'))
        city = self.translate(self.context.get('city'))
        country = self.translate(self.context.get('country'))

        return f'{name}<br/><em>{address1}<br/>{address2}<br/>{zipcode} - {city}<br/>{country}</em><br/><br/><strong>{vat_number}</strong>'

    def wrap(self, *args) -> tuple:
        return self.max_width, self.max_height

    def draw(self) -> None:
        self.canv.saveState()
        # draw box outline
        self.canv.setFillColor(self.fill_color)
        self.canv.setStrokeColor(self.stroke_color)
        self.canv.rect(0, 0, self.max_width, self.max_height, fill=1)

        # draw address paragraph
        if self.context is not None:
            p = Paragraph(self.format_data(), self.default_left_style)
            available_width = self.max_width - 2 * mm
            available_height = self.max_height - 2 * mm
            tw, th = p.wrap(available_width, available_height)
            p.drawOn(self.canv, 2 * mm, self.max_height - th - 2 * mm)

            # draw qr code
            qr_data = self.context.get('qr_data', 'no-data-found')
            qr_code = InvoiceQrCode(qr_data)
            w, h = qr_code.wrap(self.width, self.height)
            px = available_width - w - 0*mm
            py = available_height - h - 1*mm
            qr_code.drawOn(self.canv, px, py)

        # restore canvas state
        self.canv.restoreState()

