from reportlab.lib.units import mm
from reportlab.pdfgen.canvas import Canvas
from reportlab.platypus import Flowable, BaseDocTemplate

from .invoice_items_box import InvoiceItemsBox
from .invoice_customer_box import InvoiceCustomerBox
from .invoice_dates_box import InvoiceDatesBox
from .invoice_page_template import InvoicePageTemplate
from .invoice_summary_box import InvoiceSummaryBox
from .invoice_title_box import InvoiceTitleBox


class InvoiceHeader(Flowable):
    default_width = 186 * mm
    default_height = 50 * mm

    def __init__(self, context, lang='en', *args, **kwargs) -> None:
        super().__init__()
        self.lang = lang
        self.data = context
        self.customer_data = context.get('customer_data', {})
        self.invoice_data = context.get('invoice_data', {})
        self.width = self.default_width
        self.height = self.default_height

    def wrap(self, avail_width: float, avail_height: float, *args) -> tuple:
        self.width = avail_width
        return self.width, self.height

    def draw(self) -> None:
        self.canv.saveState()
        # calculates size and position of header boxes
        self.twin_boxes_w = (self.width - 5.0 * mm) / 2.0
        self.twin_boxes_h = 45 * mm
        twin_boxes_y = self.height - self.twin_boxes_h - 50 * mm
        lbox_x = 0
        rbox_x = self.width - self.twin_boxes_w
        self.box1_rect = (lbox_x, twin_boxes_y, self.twin_boxes_w, self.twin_boxes_h)
        self.box2_rect = (rbox_x, twin_boxes_y, self.twin_boxes_w, self.twin_boxes_h)

        # draw customer box
        customer_box = InvoiceCustomerBox(data=self.customer_data, lang=self.lang, max_width=self.twin_boxes_w,
                                          max_height=self.twin_boxes_h)
        w, h = customer_box.wrap(self.width, self.height)
        px = 0
        py = self.height - self.twin_boxes_h
        customer_box.drawOn(self.canv, px, py)

        # draw title box
        title_box = InvoiceTitleBox(context=self.invoice_data, lang=self.lang, max_width=self.twin_boxes_w,
                                    max_height=self.twin_boxes_h)
        w, h = title_box.wrap(self.width, self.height)
        px = rbox_x
        py = self.height - self.twin_boxes_h
        title_box.drawOn(self.canv, px, py)

        # draw dates table
        dates_box_width = self.twin_boxes_w / 1.8
        dates_box = InvoiceDatesBox(context=self.invoice_data, max_width=dates_box_width, lang=self.lang)
        w, h = dates_box.wrap(self.width, self.height)
        px = self.width - self.twin_boxes_w
        py = self.height - h
        dates_box.drawOn(self.canv, px, py)

        # draw summary table
        summary_box = InvoiceSummaryBox(context=self.invoice_data, max_width=self.twin_boxes_w, lang=self.lang)
        w, h = summary_box.wrap(self.width, self.height)
        px = self.width - self.twin_boxes_w
        py = self.height - self.twin_boxes_h
        summary_box.drawOn(self.canv, px, py)

        self.canv.restoreState()


class InvoiceDoc(BaseDocTemplate):

    def __init__(self, *args, **kwargs) -> None:
        self.lang = 'en'
        kwargs['leftMargin'] = 12 * mm
        kwargs['rightMargin'] = 12 * mm
        kwargs['topMargin'] = 10 * mm
        kwargs['bottomMargin'] = 10 * mm
        BaseDocTemplate.__init__(self, *args, **kwargs)

    def build(self, context: dict, lang: str = 'en', canvasMaker=Canvas) -> None:
        self._calc()

        issuer_data = context.get('issuer_data', {})
        invoice_page_template = InvoicePageTemplate(
            context=issuer_data,
            lang=lang,
            pagesize=self.pagesize
        )
        self.addPageTemplates(invoice_page_template)

        story = []
        # invoice header
        header = InvoiceHeader(context, lang)
        story.append(header)

        # invoice items (table)
        invoice_items = context.get('invoice_items', [])
        story.append(InvoiceItemsBox(data=invoice_items, lang=lang))

        BaseDocTemplate.build(self, story)
