from datetime import date

from django.db import models
from django.db.models import F, Q, Case, When
from django.db.models.functions import Length


class QueryCmsLegalDoc(models.QuerySet):
    default_language = 'en'
    table_prefix_by_language = {
        'ca': 'catalan',
        'en': 'english',
        'es': 'spanish',
        'fr': 'french',
        'it': 'italian',
        'pt': 'portuguese',
    }

    def _get_i18n_table_prefix(self, language: str) -> str:
        current_lang = language.lower()
        default_lang = self.default_language.lower()
        return self.table_prefix_by_language.get(current_lang) or self.table_prefix_by_language.get(default_lang)

    def with_date_within_valid_range(self, date_value: date) -> models.QuerySet:
        start_date_is_valid = Q(valid_from__lte=date_value)
        end_date_is_valid = (Q(valid_until__gte=date_value) | Q(valid_until__isnull=True))
        date_is_into_valid_range = start_date_is_valid & end_date_is_valid
        return self.filter(date_is_into_valid_range)

    def with_translated_title_or_default(self, language: str) -> models.QuerySet:
        # Empty fields are replaced at database level with default value
        current_i18n_prefix = self._get_i18n_table_prefix(language)
        default_i18n_prefix = self._get_i18n_table_prefix(self.default_language)

        i18n_title_is_empty = (
                Q(**{f'{current_i18n_prefix}__title__isnull': True}) |
                Q(**{f'{current_i18n_prefix}__title__exact': ''})
        )
        i18n_title = Case(
            When(i18n_title_is_empty, then=F(f'{default_i18n_prefix}__title')),
            default=F(f'{current_i18n_prefix}__title')
        )
        return self.annotate(title=i18n_title)

    def with_translated_copyright_notice_or_default(self, language: str) -> models.QuerySet:
        # Empty fields are replaced at database level with default value
        current_i18n_prefix = self._get_i18n_table_prefix(language)
        default_i18n_prefix = self._get_i18n_table_prefix(self.default_language)

        i18n_copyright_notice_is_empty = (
                Q(**{f'{current_i18n_prefix}__copyright_notice__isnull': True}) |
                Q(**{f'{current_i18n_prefix}__copyright_notice__exact': ''})
        )
        i18n_copyright_notice = Case(
            When(i18n_copyright_notice_is_empty, then=F(f'{default_i18n_prefix}__copyright_notice')),
            default=F(f'{current_i18n_prefix}__copyright_notice')
        )
        return self.annotate(copyright_notice=i18n_copyright_notice)

    def with_translated_html_content_or_default(self, language: str) -> models.QuerySet:
        # Empty fields are replaced at database level with default value
        current_i18n_prefix = self._get_i18n_table_prefix(language)
        default_i18n_prefix = self._get_i18n_table_prefix(self.default_language)

        i18n_html_content_is_null = (
                Q(**{f'{current_i18n_prefix}__html_content__isnull': True}) |
                Q(**{f'{current_i18n_prefix}__html_content__exact': ''})
        )
        i18n_html_content = Case(
            When(i18n_html_content_is_null, then=F(f'{default_i18n_prefix}__html_content')),
            default=F(f'{current_i18n_prefix}__html_content')
        )
        return self.annotate(html_content=i18n_html_content)

    def with_translated_html_content_size_or_default(self, language: str) -> models.QuerySet:
        # Empty fields are replaced at database level with default value
        current_i18n_prefix = self._get_i18n_table_prefix(language)
        default_i18n_prefix = self._get_i18n_table_prefix(self.default_language)

        i18n_html_content_is_null = (
                Q(**{f'{current_i18n_prefix}__html_content__isnull': True}) |
                Q(**{f'{current_i18n_prefix}__html_content__exact': ''})
        )
        i18n_html_content_size = Case(
            When(i18n_html_content_is_null, then=Length(f'{default_i18n_prefix}__html_content')),
            default=Length(f'{current_i18n_prefix}__html_content')
        )
        return self.annotate(html_content_size=i18n_html_content_size)
