

class AirplaneKeyboard {
    domNamespace = "http://www.w3.org/2000/svg";

    defaultSvgX = 250;
    defaultSvgY = 60;
    defaultSvgWidth = 800;
    defaultSvgHeight = 180;
    defaultSvgBackgroundColor = "green";

    defaultMaxHorizontalGrid = 32;
    defaultMaxVerticalGrid = 10;
    defaultLayoutSettings = "";

    defaultOnMouseDownHandler = (button) => { alert(`AirplaneKeyboardClass: clicked on  ${button.uid}`); };
    defaultButtonClass = AirplaneButton;

    constructor(settings={}) {
        /* dependencies injection */
        this.buttonClass = (typeof(settings.buttonClass)=='function') ? settings.buttonClass : this.defaultButtonClass;

        /* other settings */
        this.svgX = (typeof(settings.svgX)=='number') ? settings.svgX : this.defaultSvgX;
        this.svgY = (typeof(settings.svgY)=='number') ? settings.svgY : this.defaultSvgY;
        this.svgWidth = (typeof(settings.svgWidth)=='number') ? settings.svgWidth : this.defaultSvgWidth;
        this.svgHeight = (typeof(settings.svgHeight)=='number') ? settings.svgHeight : this.defaultSvgHeight;

        this.viewBox = `${0} ${0} ${this.svgWidth} ${this.svgHeight}`;
        this.horizontalGridPitch = this.svgWidth / this.defaultMaxHorizontalGrid;

        this.onMouseDownHandler = this.defaultOnMouseDownHandler;

        this.domNode = this.domParentNodeSetup();
    }

    domParentNodeSetup() {
        let node = document.createElementNS(this.domNamespace, "svg");
        node.setAttribute('viewBox', this.viewBox);
        node.setAttribute('x', this.svgX);
        node.setAttribute('y', this.svgY);
        node.setAttribute('width', this.svgWidth);
        node.setAttribute('height', this.svgHeight);
        return node;
    }

    domChildrenNodesSetup(parsedData, parentNode) {
        parentNode.replaceChildren();
        for (let dataRow of parsedData) {
            for (let dataCell of dataRow) {
                let button = new this.buttonClass(dataCell);
                button.setupClickHandler(this.onMouseDownHandler);
                parentNode.appendChild(button.domNode);
            }
        }
    }

    setupClickHandler(hookFunction) {
        if (typeof(hookFunction) == 'function') {
            this.onMouseDownHandler = hookFunction;
        }
    }

    setupLayout(uncleanedSettings) {
        let cleanedSettings = this.cleanSettings(uncleanedSettings);
        let parsedLayout = this.parseKeyboardGridSettings(cleanedSettings, this.horizontalGridPitch, this.svgHeight);
        this.domChildrenNodesSetup(parsedLayout, this.domNode);
        return cleanedSettings;
    }

    cleanSettings(uncleanedSettings) {
        let text = uncleanedSettings.replaceAll(' ', '');
        text = text.toUpperCase();
        let textRows = text.split('\n');
        textRows.splice(this.defaultMaxHorizontalGrid);

        let cleanedText = "";
        for (let index=0; index<textRows.length; index++) {
            let rowValue = textRows[index];
            let cleanedRow = rowValue.slice(0, this.defaultMaxVerticalGrid);
            cleanedText += (index > 0) ? `\n${cleanedRow}` : cleanedRow;
        }
        return cleanedText;
    }

    parseKeyboardGridSettings(cleanedSettings, horizontalGridPitch, verticalPixels) {
        let rowsData = cleanedSettings.split('\n');

        let parsedData = [];
        let nonEmptyRowsCounter = 0;
        for (let rowIndex=0; rowIndex < rowsData.length; rowIndex++) {
            let rowData = rowsData[rowIndex];

            if (rowData.length) {
                nonEmptyRowsCounter++;
            }
            let rowName = nonEmptyRowsCounter.toString();
            let answer = this.parseKeyboardRowSettings(
                rowIndex,
                rowName,
                rowData,
                horizontalGridPitch,
                verticalPixels
            );
            parsedData.push(answer);
        }
        return parsedData;
    }

    parseKeyboardRowSettings(rowIndex, rowName, rowData, horizontalGridPitch, verticalPixels) {
        let rowLength = rowData.length;
        let verticalGridPitch = verticalPixels / rowLength;

        let parsedData = [];
        for (let buttonIndex=0; buttonIndex < rowLength; buttonIndex++) {
            let buttonName = rowData[buttonIndex];
            if (buttonName != '-') {
                let data = this.parseSingleButtonSettings(
                    rowIndex,
                    rowName,
                    buttonIndex,
                    buttonName,
                    horizontalGridPitch,
                    verticalGridPitch
                );
                parsedData.push(data);
            }
        }
        return parsedData;
    }

    parseSingleButtonSettings(rowIndex, rowName, buttonIndex, buttonName, horizontalGridPitch, verticalGridPitch) {
        return {
            uid: this.makeButtonUID(rowName, buttonName),
            x: rowIndex * horizontalGridPitch,
            y: buttonIndex * verticalGridPitch,
            width: horizontalGridPitch,
            height: verticalGridPitch,
        };
    }

    makeButtonUID(rowName, buttonName) {
        return `${rowName}-${buttonName}`;
    }
}
